/* ==================================================================
File        : debris.hlsl
Author      : Hlynur Tryggvason
Date        : 12/12/2008
Format      : HLSL/Cg
Description : Next gen debris shaders.

================================================================= */

#include "VertexShaderSemantics.fx"
#include "FragmentShaderSemantics.fx"

#include "fog.hlsl"

#ifdef CG_PS3
	#define	_PS_POSITION		POSITION
	#define _PS_COLOR			COLOR
	#define _PS_TEXCOORD0		TEXCOORD0
	#define _PS_TEXCOORD1		TEXCOORD1
	#define _PS_TEXCOORD2		TEXCOORD2
	#define _PS_TEXCOORD3		TEXCOORD3
	#define _PS_TEXCOORD4		TEXCOORD4

#else

	#define	_PS_POSITION		POSITION
	#define _PS_COLOR			COLOR

	#ifdef SM1
		#define _PS_TEXCOORD0		TEXCOORD0
		#define _PS_TEXCOORD1		TEXCOORD1
		#define _PS_TEXCOORD2		TEXCOORD2
		#define _PS_TEXCOORD3		TEXCOORD3
		#define _PS_TEXCOORD4		TEXCOORD4
	#else
		#define _PS_TEXCOORD0		TEXCOORD0_centroid
		#define _PS_TEXCOORD1		TEXCOORD1_centroid
		#define _PS_TEXCOORD2		TEXCOORD2_centroid
		#define _PS_TEXCOORD3		TEXCOORD3_centroid
		#define _PS_TEXCOORD4		TEXCOORD4_centroid
	#endif
#endif

#define g_ViewportScaleBias	vs_viewportScaleBias

//@:-texformat layer1_sampler RGBA8
//@:-texformat layer2_sampler RGBA8
sampler2D layer0_sampler : register(s0);
sampler2D layer1_sampler : register(s1);
sampler2D layer2_sampler : register(s2);

// Register 0 sampler
// - debris_px		: Particle diffuse texture
// - debrisSoft_px	: Particle diffuse texture
// - debrisGlass_px	: Backbuffer
#define g_DiffuseTex	layer0_sampler
#define g_ColorBuffer	layer0_sampler

// Register 1 sampler
// - debris_px		: Not used
// - debrisSoft_px	: Depth buffer
// - debrisGlass_px	: Distortion texture
#define g_DepthBuffer	layer1_sampler
#define g_DistortTex	layer1_sampler

// Register 2 sampler
// - debris_px		: Normalmap
// - debrisSoft_px	: Normalmap
// - debrisGlass_px	: Not used
#define g_NormalTex		layer2_sampler

struct VS_IN {
	float4 position		: POSITION;
	float3 uvSet0		: TEXCOORD0;	// xy : UV texture coords, z : particle "radius" for soft particles
	float4 colorSet0	: COLOR;	
};

struct VS_TRAILS_IN {
	float4 position		: POSITION;
	float3 uvSet0		: TEXCOORD0;	// xy : UV texture coords, z : particle "radius" for soft particles
	float4 colorSet0	: COLOR;
	float3 uvSet2		: TEXCOORD1;	// xyz : World position of next vertex in the curve
};

struct VS_OUT { 
	float4 position		: _PS_POSITION;
	float4 texcoord		: _PS_TEXCOORD0;	// xy : UV texture coords, z : fog factor
#if defined(_LIT)
	float4 eyevec		: _PS_TEXCOORD1;
#endif
	float4 color		: _PS_COLOR;	
};

struct PS_IN {
	float4 color		: _PS_COLOR;
	float4 texcoord		: _PS_TEXCOORD0;
#if defined(_LIT)
	float4 eyevec		: _PS_TEXCOORD1;
#endif
};

struct VS_OUT_SOFT
{
	float4 position		: _PS_POSITION;
	float4 color		: _PS_TEXCOORD0;
	float4 texcoord		: _PS_TEXCOORD1;	// xy : particle texture,		zw : projection parameters
	float4 softpos		: _PS_TEXCOORD2;	// xy : screen UV coordinate,	zw : the fragment depth with w value
#if defined(_LIT)
	float4 eyevec		: _PS_TEXCOORD3;
#endif	
	float2 fadeScaleFog	: _PS_TEXCOORD4;	// x : fade scale, y : fog factor
};

struct PS_IN_SOFT
{	
	float4 color		: _PS_TEXCOORD0;
	float4 texcoord		: _PS_TEXCOORD1;
	float4 softpos		: _PS_TEXCOORD2;	// xy : screen UV coordinate, zw : the fragment depth with w value
#if defined(_LIT)
	float4 eyevec		: _PS_TEXCOORD3;
#endif
	float2 fadeScaleFog	: _PS_TEXCOORD4;	// x : fade scale, y : fog factor
};

// --------------------------------------------
// Utility function that rotates a vector using the provided matrix
// --------------------------------------------
inline float4 rotateVector(float4 v0, float4x4 m)
{
	float4 v;
	v.x = (v0.x * m[0][0]) + (v0.y * m[1][0]) + (v0.z * m[2][0]);
	v.y = (v0.x * m[0][1]) + (v0.y * m[1][1]) + (v0.z * m[2][1]);
	v.z = (v0.x * m[0][2]) + (v0.y * m[1][2]) + (v0.z * m[2][2]);
	v.w = v0.w;

	return v;
}

// --------------------------------------------
// Utility functions for fog calculation
// --------------------------------------------

inline float3 computeEyePosition(float4 p)
{
	float4 eyePositionH = mul(p, vs_view);

	// we can assume .w == 1 in the general case
	return eyePositionH.xyz;
}

// --------------------------------------------
// Utility functions that calculates and lights the rgba colour of the particle
// --------------------------------------------

inline float4 calculateSurfaceColour( float2 uv, float4 vtxcol )
{
	return tex2D(g_DiffuseTex, uv) * vtxcol * 2;
}

inline float4 fogStage( float4 col, half fogFactor )
{
#if defined(_FOG)
	return float4(lerp(fog_color.rgb, col.rgb, fogFactor),col.a);
#else
	return col;
#endif
}

inline float4 calculateColour( float2 uv, float4 vtxcol, half fogFactor )
{
	float4 col = calculateSurfaceColour( uv, vtxcol );
	
	return fogStage(col, fogFactor);
}

inline float4 calculateLitColour( float2 uv, float4 vtxcol, float4 eyevec, half fogFactor )
{
	//
	// Cheap scattering effect. Evaluates to zero if looking into the light
	//	
	float scattering0 = 1.0f - saturate(max(0, dot(eyevec, lightPosition0)));
	float scattering1 = 1.0f - saturate(max(0, dot(eyevec, lightPosition1)));
	float scattering2 = 1.0f - saturate(max(0, dot(eyevec, lightPosition2)));
	
#if defined(_NORMALS)	
	//
	// Use normal maps
	//
	float4 normal = tex2D(g_NormalTex, uv);	
	normal = normal * 2.0 - 1.0;	
	
	// Invert z and fix w
	normal.z = -normal.z;
	normal.w = 1.0f;

	float4 diffuseLight = lightColor0 * dot(normal, lightPosition0) * scattering0;
	diffuseLight += lightColor1 * dot(normal, lightPosition1) * scattering1;	
	diffuseLight += lightColor2 * dot(normal, lightPosition2) * scattering2;
#else
	//
	// Add scatter factor
	//
	float4 diffuseLight = lightColor0 * scattering0;
	diffuseLight += lightColor1 * scattering1;	
	diffuseLight += lightColor2 * scattering2;
#endif
	
	// Add ambient	
	diffuseLight = (1.0f - diffuseLight) * sceneAmbientColor + diffuseLight;
	diffuseLight.a = 1.0f;

	// Calculate final colour
	float4 col = calculateSurfaceColour( uv, vtxcol ) * diffuseLight;
	return fogStage(col, fogFactor);
}



// --------------------------------------------
// Simple particles
// --------------------------------------------

inline VS_OUT debrisCommon_vx(float4 pos, float3 uvset, float4 colour)
{
	VS_OUT Out;

	Out.color = colour;
	Out.texcoord.xy = uvset;

#if defined(_FOG)
	float3 eyePosition = computeEyePosition(pos);
	Out.texcoord.z = vs_fogVertexPlanarExponential2(pos, eyePosition);
	Out.texcoord.w = 1;
#else
	Out.texcoord.zw = 1;	
#endif

#if defined(_LIT)
	Out.eyevec = normalize(rotateVector(pos - worldCamPos, vs_view));
#endif

	return Out;
}

VS_OUT debris_vx(VS_IN input)
{
	VS_OUT Out = debrisCommon_vx(input.position, input.uvSet0, input.colorSet0);
	Out.position = mul(input.position, viewProj);

	return Out;
}

VS_OUT debrisTrails_vx(VS_TRAILS_IN input)
{
	VS_OUT Out = debrisCommon_vx(input.position, input.uvSet0, input.colorSet0);

	// Calculate screen space positions of the incoming vertex
	float4 v0 = mul(input.position, viewProj);

	// Calculate screen space position of the next vertex in the curve
	float4 v1 = mul( float4(input.uvSet2, 1.0f), viewProj);

	// Calculate a vector in screen space to the next vertex	
	float2 vectorToNext = (v1 - v0).xy;

	// Normalize		
	float distToNext = length(vectorToNext);		
	vectorToNext /= distToNext;	

	// Attempt to fix overlaps in curve by slightly fading out 
	// vertices that are close to another vertex
	Out.color.a *= saturate( pow(distToNext / 0.1f, 3.0f) );	

	// Create tangent
	float2 tangent = float2( -vectorToNext.y, vectorToNext.x );

	// Scale the tangent
	tangent *= input.uvSet0.z; 

	// Use the tangent to offset the vertex
	v0.xy += tangent.xy;	

	Out.position = v0;	

	return Out;
}

float4 debris_px(PS_IN In) : COLOR
{	
#if defined(_LIT)
	return calculateLitColour( In.texcoord.xy, In.color, In.eyevec, In.texcoord.z );
#else
	return calculateColour( In.texcoord.xy, In.color, In.texcoord.z );
#endif
}

// --------------------------------------------
// Soft particles (depth fading)
// --------------------------------------------

// Helper methods for depth texture sampling
#if defined(CG_PS3)
	inline float readDepthTexture(sampler2D tex, half2 uvTex)
	{
		float4 depthColor = tex2D(tex, uvTex).rgba;
		float4 depthFactor = float4(256.0/16777215.0, 1.0/16777215.0, 0.0f, 256.0*256.0/16777215.0);
		return dot( round(depthColor*255.0), depthFactor );
	}
#elif defined(_WINPC)
	#if defined(_Z_RGB)
		inline float readDepthTexture(sampler2D tex, half2 uvTex)
		{
			float4 depthColor = tex2D(tex, uvTex).rgba;
			float3 depthFactor = float3(256.0*256.0/16777215.0, 256.0/16777215.0, 1.0/16777215.0);
			return dot(round(depthColor.rgb*255.0), depthFactor);
		}
	#elif defined(_Z_FLOAT)
		inline float readDepthTexture(sampler2D tex, half2 uvTex)
		{
			return tex2D(tex, uvTex).r;						
		}
	#else // _Z_ARG
		inline float readDepthTexture(sampler2D tex, half2 uvTex)
		{
			float4 depthColor = tex2D(tex, uvTex).rgba;
			float3 depthFactor = float3(256.0*256.0/16777215.0, 256.0/16777215.0, 1.0/16777215.0);
			return dot(round(depthColor.arg*255.0), depthFactor);
		}
	#endif
#else // 360
	inline float readDepthTexture(sampler2D tex, half2 uvTex)
	{
	#ifdef USE_INVERTED_PROJECTION_MTX
		return 1.0 - tex2D(tex, uvTex).r;
	#else
		return tex2D(tex, uvTex).r;
	#endif
	}
#endif


// Soft particles vertex shader
VS_OUT_SOFT debrisSoft_vx( VS_IN input )
{
	VS_OUT_SOFT Out;
	Out.position = mul(input.position, viewProj);
	Out.color = input.colorSet0;
	Out.texcoord.xy = input.uvSet0;
	Out.texcoord.z = vs_projection_params.x;
	Out.texcoord.w = vs_projection_params.w;
	
	// Pass the particle position to the pixel shader
	Out.softpos = Out.position;
	Out.softpos.xy = (Out.position.xy/Out.position.ww) * float2(0.5,-0.5) + float2(0.5,0.5);
	Out.softpos.xy = Out.softpos.xy * g_ViewportScaleBias.zw + g_ViewportScaleBias.xy;

	// Store the size of the quad
	Out.fadeScaleFog.x = 0.7f / input.uvSet0.z;

#if defined(_FOG)
	float3 eyePosition = computeEyePosition(input.position);
	Out.fadeScaleFog.y = vs_fogVertexPlanarExponential2(input.position, eyePosition);	
#else
	Out.fadeScaleFog.y = 1;	
#endif

#if defined(_LIT)
	Out.eyevec = normalize(rotateVector(input.position - worldCamPos, vs_view));
#endif

	return Out;
}

// Soft particles pixel shader
float4 debrisSoft_px(PS_IN_SOFT In) : COLOR
{	
	// Sample the particle's diffuse texture
#if defined(_LIT)
	float4 outColour = calculateLitColour( In.texcoord.xy, In.color, In.eyevec, In.fadeScaleFog.y);	
#else
	float4 outColour = calculateColour( In.texcoord.xy, In.color, In.fadeScaleFog.y );
#endif
	
	// Sample depth buffer
	float zBufferVal = readDepthTexture(g_DepthBuffer, In.softpos.xy);

	// Grab particle z location and the z value in the depth buffer
	float2 zValues = float2(In.softpos.z / In.softpos.w, zBufferVal);
	
	// Put z values in linear space
	zValues = In.texcoord.z / (1.0 - zValues * In.texcoord.w);	

	// Modify alpha value based on z values
	outColour.a *= saturate((zValues.y - zValues.x) * In.fadeScaleFog.x);
	
	return outColour;
}

// --------------------------------------------
// Glass particles (background distortion)
// --------------------------------------------

VS_OUT debrisGlass_vx( VS_IN input )
{
	VS_OUT Out;

	Out.position = mul(input.position, viewProj);
	//Out.oPosition = mul(input.position, worldViewProj); // This was on PS3
	Out.color = input.colorSet0;

	Out.texcoord.xy = (Out.position.xy/Out.position.ww) * float2(0.5,-0.5) + float2(0.5,0.5);
	#if defined(_360_TARGET)
		Out.texcoord.xy = Out.texcoord.xy * g_ViewportScaleBias.zw + g_ViewportScaleBias.xy;
	#endif
	Out.texcoord.zw = input.uvSet0;

	return Out;
}

float4 debrisGlass_px( PS_IN In ) : COLOR
{
	float4 offset = (tex2D(g_DistortTex, In.texcoord.zw) - float4(0.5,0.5,0.0,0.0)) * float4(0.01,0.01,1.0,1.0);
	float4 oColour = float4(tex2D(g_ColorBuffer, In.texcoord.xy + offset.xy).rgb, 1.0);
	oColour *= In.color * 2;
	oColour.a *= offset.z;
	
	return oColour;
}